// ============ IMPORTS ============
import { registerOverlay, init as klinechartsInit } from 'klinecharts';
import { KLineChartPro } from '@klinecharts/pro';
import watermarkUrl from './watermark.png';
import saveIconUrl from './save.png';
import themeIconUrl from './light.png';
import './themePresets.js';
import './chartSettingsModel.js';
import './settingsPanelView.js';
import './chartIntercept.js';
import '@klinecharts/pro/dist/klinecharts-pro.css';

// ============ UTILITIES ============
function periodToInterval(period) {
    var t = (period && period.text) ? period.text : '15m';
    var m = {
        '1m': '1m', '5m': '5m', '15m': '15m',
        '1h': '1h', '1H': '1h', '2h': '2h', '2H': '2h', '4h': '4h', '4H': '4h',
        '1d': '1d', 'D': '1d', '1w': '1w', 'W': '1w', '1M': '1M', 'M': '1M'
    };
    return m[t] || '15m';
}

var STATIC_DEFAULT_TICKERS = [
    'BTCUSDT', 'ETHUSDT', 'BNBUSDT', 'XRPUSDT', 'DOGEUSDT',
    'ADAUSDT', 'SOLUSDT', 'MATICUSDT', 'DOTUSDT', 'AVAXUSDT',
    'LINKUSDT', 'UNIUSDT', 'LTCUSDT', 'ATOMUSDT', 'FILUSDT'
];

function toSymbolInfo(ticker) {
    return {
        ticker: ticker,
        shortName: ticker,
        exchange: 'BINANCE',
        market: 'crypto',
        priceCurrency: 'usd'
    };
}

function getDefaultSymbolList() {
    return STATIC_DEFAULT_TICKERS.map(toSymbolInfo);
}

function searchSymbolsForUI(query) {
    if (!query || String(query).length < 2) {
        return Promise.resolve(getDefaultSymbolList());
    }
    var q = String(query).toUpperCase();
    return fetch('https://api.binance.com/api/v3/exchangeInfo')
        .then(function (r) { return r.json(); })
        .then(function (data) {
            var symbols = (data.symbols || []).filter(function (s) {
                return s.symbol && s.symbol.indexOf(q) !== -1 && s.symbol.endsWith('USDT') && s.status === 'TRADING';
            }).slice(0, 10).map(function (s) { return toSymbolInfo(s.symbol); });
            return symbols.length ? symbols : getDefaultSymbolList();
        })
        .catch(function () { return getDefaultSymbolList(); });
}

// ============ DATAFEED & BINANCE ============
function BinanceDatafeed() {
    this._ws = null;
    this._subscribeCallback = null;
    this._abortController = null;
}
BinanceDatafeed.prototype.searchSymbols = function () {
    return Promise.resolve(getDefaultSymbolList());
};
BinanceDatafeed.prototype.getHistoryKLineData = function (symbol, period, from, to) {
    var self = this;
    var ticker = (symbol && symbol.ticker) ? symbol.ticker : 'BTCUSDT';
    var interval = periodToInterval(period);
    if (self._abortController) self._abortController.abort();
    self._abortController = new AbortController();
    var url = '/api/klines?symbol=' + encodeURIComponent(ticker) + '&interval=' + encodeURIComponent(interval) + '&limit=1000';
    if (from != null) url += '&startTime=' + from;
    if (to != null) url += '&endTime=' + to;
    return fetch(url, { signal: self._abortController.signal })
        .then(function (r) {
            if (!r.ok) return [];
            return r.json();
        })
        .then(function (rows) {
            if (!Array.isArray(rows)) return [];
            return rows.map(function (d) {
                // Backend proxy returns {timestamp, open, high, low, close, volume} (KLineChart Pro format)
                return {
                    timestamp: Number(d.timestamp),
                    open: Number(d.open),
                    high: Number(d.high),
                    low: Number(d.low),
                    close: Number(d.close),
                    volume: Number(d.volume)
                };
            });
        })
        .catch(function (err) {
            if (err.name === 'AbortError') return [];
            return [];
        });
};
BinanceDatafeed.prototype.subscribe = function (symbol, period, callback) {
    var self = this;
    self._subscribeCallback = callback;
    var ticker = (symbol && symbol.ticker) ? symbol.ticker : 'BTCUSDT';
    var timeframe = periodToInterval(period);
    function send() {
        self._ws.send(JSON.stringify({ type: 'change_symbol', symbol: ticker }));
        self._ws.send(JSON.stringify({ type: 'change_timeframe', timeframe: timeframe }));
    }
    if (self._ws && self._ws.readyState === WebSocket.OPEN) {
        _drawingsWs = self._ws;
        send();
        return;
    }
    self._ws = new WebSocket('ws://localhost:4000');
    _drawingsWs = self._ws;
    self._ws.onopen = function () {
        send();
    };
    self._ws.onmessage = function (event) {
        try {
            var data = JSON.parse(event.data);
            if (data && data.type === 'drawings_snapshot') {
                if (typeof _onDrawingsSnapshot === 'function') _onDrawingsSnapshot(data);
                return;
            }
            if (typeof data.timestamp === 'number' && typeof data.open === 'number' && typeof data.high === 'number' && typeof data.low === 'number' && typeof data.close === 'number' && typeof data.volume === 'number') {
                var klineData = { timestamp: data.timestamp, open: data.open, high: data.high, low: data.low, close: data.close, volume: data.volume };
                if (typeof self._subscribeCallback === 'function') self._subscribeCallback(klineData);
            }
        } catch (e) {}
    };
    self._ws.onerror = function () { self._ws = null; _drawingsWs = null; };
    self._ws.onclose = function () { self._ws = null; _drawingsWs = null; };
};
BinanceDatafeed.prototype.unsubscribe = function (symbol, period) {
    this._subscribeCallback = null;
    // WebSocket must stay open so the backend does not think there are no clients and shut down.
    // Symbol/timeframe switching is done by sending change_symbol/change_timeframe on the same connection (as in v2).
};

// ============ GLOBALS ============
var chartContainer = document.getElementById('container');
var symbolSearchPanel = document.getElementById('symbol-search-panel');
var symbolSearchInput = document.getElementById('symbol-search-input');
var symbolSearchList = document.getElementById('symbol-search-list');

var WATCHLIST_STORAGE_KEY = 'watchlist';
var watchlistSymbols = [];
var watchlistSortMode = 'chronological';
var watchlistLastPrices = {};
var watchlistPriceIntervalId = null;
var currentSymbol = 'BTCUSDT';
var _pendingSymbol = null;
var _switchTimer = null;
var overlayIdsBySymbol = {};
var _drawingsWs = null;
var _onDrawingsSnapshot = null;
var _skipDrawingDelete = false;
var _loadingDrawingsSnapshot = false;
var _pendingScrollToRealTime = false;

var _symbolPanelOutsideClick = null;
var _symbolPanelEscape = null;

// ============ WATCHLIST ============
function loadWatchlistFromStorage() {
    try {
        var raw = localStorage.getItem(WATCHLIST_STORAGE_KEY);
        if (raw) {
            var arr = JSON.parse(raw);
            if (Array.isArray(arr)) {
                watchlistSymbols = arr;
                return;
            }
        }
    } catch (e) {}
    watchlistSymbols = ['BTCUSDT', 'ETHUSDT', 'BNBUSDT', 'SOLUSDT'];
}

function saveWatchlistToStorage() {
    try {
        localStorage.setItem(WATCHLIST_STORAGE_KEY, JSON.stringify(watchlistSymbols));
    } catch (e) {}
}

function addToWatchlist(symbol) {
    var s = String(symbol || '').toUpperCase().trim();
    if (!s || watchlistSymbols.indexOf(s) !== -1) return;
    watchlistSymbols.push(s);
    saveWatchlistToStorage();
    renderWatchlist();
    startWatchlistPricePolling();
}

function removeFromWatchlist(symbol) {
    watchlistSymbols = watchlistSymbols.filter(function (x) { return x !== symbol; });
    saveWatchlistToStorage();
    renderWatchlist();
    startWatchlistPricePolling();
}

function formatWatchlistPrice(priceStr) {
    if (priceStr == null || priceStr === '' || priceStr === '\u2014') return '\u2014';
    var n = parseFloat(priceStr);
    if (Number.isNaN(n)) return priceStr;
    return n >= 1 ? n.toFixed(2) : n.toFixed(4);
}

function updateWatchlistPriceDisplays() {
    document.querySelectorAll('.watchlist-item').forEach(function (el) {
        var symEl = el.querySelector('.watchlist-symbol');
        var priceEl = el.querySelector('.watchlist-price');
        if (symEl && priceEl) priceEl.textContent = formatWatchlistPrice(watchlistLastPrices[symEl.textContent]);
    });
}

function startWatchlistPricePolling() {
    if (watchlistPriceIntervalId) {
        clearInterval(watchlistPriceIntervalId);
        watchlistPriceIntervalId = null;
    }
    if (watchlistSymbols.length === 0) return;
    var poll = function () {
        if (watchlistSymbols.length === 0) return;
        var symbolsParam = encodeURIComponent(JSON.stringify(watchlistSymbols));
        fetch('https://api.binance.com/api/v3/ticker/price?symbols=' + symbolsParam)
            .then(function (r) { return r.json(); })
            .then(function (data) {
                if (Array.isArray(data)) {
                    data.forEach(function (o) {
                        if (o && o.symbol != null) watchlistLastPrices[o.symbol] = o.price;
                    });
                    updateWatchlistPriceDisplays();
                }
            })
            .catch(function () {});
    };
    poll();
    watchlistPriceIntervalId = setInterval(poll, 1000);
}

function renderWatchlist() {
    var container = document.getElementById('watchlist-content');
    if (!container) return;
    container.innerHTML = '';
    var sorted = watchlistSymbols.slice();
    if (watchlistSortMode === 'a-z') sorted.sort();
    else if (watchlistSortMode === 'z-a') sorted.sort().reverse();
    sorted.forEach(function (symbol) {
        var item = document.createElement('div');
        item.className = 'watchlist-item';
        if (symbol === currentSymbol) item.classList.add('active');
        item.innerHTML = '<span class="watchlist-symbol">' + symbol + '</span><span class="watchlist-price">' + formatWatchlistPrice(watchlistLastPrices[symbol]) + '</span>';
        item.addEventListener('click', function () {
            if (symbol === currentSymbol) return;
            switchSymbol(symbol);
        });
        item.addEventListener('contextmenu', function (e) {
            e.preventDefault();
            removeFromWatchlist(symbol);
        });
        container.appendChild(item);
    });
}

function initWatchlist() {
    loadWatchlistFromStorage();
    renderWatchlist();
    var sortToggle = document.getElementById('watchlist-sort-toggle');
    if (sortToggle) {
        sortToggle.addEventListener('click', function () {
            if (watchlistSortMode === 'chronological') {
                watchlistSortMode = 'a-z';
                sortToggle.textContent = 'A-Z';
                sortToggle.classList.add('active');
            } else if (watchlistSortMode === 'a-z') {
                watchlistSortMode = 'z-a';
                sortToggle.textContent = 'Z-A';
            } else {
                watchlistSortMode = 'chronological';
                sortToggle.textContent = 'A-Z';
                sortToggle.classList.remove('active');
            }
            renderWatchlist();
        });
    }
    window.addEventListener('beforeunload', saveWatchlistToStorage);
    startWatchlistPricePolling();
}

// ============ SYMBOL SWITCHING ============
function hideSymbolSearchPanel() {
    if (symbolSearchPanel) symbolSearchPanel.classList.remove('show');
    if (_symbolPanelOutsideClick) {
        document.removeEventListener('click', _symbolPanelOutsideClick);
        _symbolPanelOutsideClick = null;
    }
    if (_symbolPanelEscape) {
        document.removeEventListener('keydown', _symbolPanelEscape);
        _symbolPanelEscape = null;
    }
}

function showSymbolSearchPanel() {
    if (!symbolSearchPanel || !symbolSearchList) return;
    symbolSearchPanel.classList.add('show');
    if (symbolSearchInput) {
        symbolSearchInput.value = '';
        symbolSearchInput.focus();
    }
    searchSymbolsForUI('').then(function (list) { renderSymbolList(list); });
    _symbolPanelOutsideClick = function (e) {
        if (symbolSearchPanel.contains(e.target)) return;
        if (chartContainer && chartContainer.contains(e.target) && e.target.closest && e.target.closest('.symbol')) return;
        hideSymbolSearchPanel();
    };
    _symbolPanelEscape = function (e) {
        if (e.key === 'Escape') hideSymbolSearchPanel();
    };
    setTimeout(function () {
        document.addEventListener('click', _symbolPanelOutsideClick);
        document.addEventListener('keydown', _symbolPanelEscape);
    }, 0);
}

function renderSymbolList(symbols) {
    if (!symbolSearchList || !chart) return;
    symbolSearchList.innerHTML = '';
    (symbols || []).forEach(function (symbolInfo) {
        var row = document.createElement('div');
        row.className = 'symbol-search-row';
        var nameEl = document.createElement('span');
        nameEl.className = 'symbol-name';
        nameEl.textContent = symbolInfo.shortName || symbolInfo.ticker;
        nameEl.title = 'Show chart';
        var actions = document.createElement('span');
        actions.className = 'row-actions';
        var rightArrow = document.createElement('span');
        rightArrow.className = 'action-add-watchlist';
        rightArrow.setAttribute('title', 'Add to watchlist');
        rightArrow.textContent = '\u2192';
        rightArrow.addEventListener('click', function (e) {
            e.stopPropagation();
            addToWatchlist(symbolInfo.ticker);
        });
        var downArrow = document.createElement('span');
        downArrow.className = 'action-show-chart';
        downArrow.setAttribute('title', 'Show chart without adding to watchlist');
        downArrow.textContent = '\u2193';
        downArrow.addEventListener('click', function (e) {
            e.stopPropagation();
            switchSymbol(symbolInfo.ticker);
        });
        actions.appendChild(rightArrow);
        actions.appendChild(downArrow);
        row.appendChild(nameEl);
        row.appendChild(actions);
        nameEl.addEventListener('click', function () {
            switchSymbol(symbolInfo.ticker);
        });
        symbolSearchList.appendChild(row);
    });
}

function switchSymbol(newSymbol) {
    if (!chart || !newSymbol) return;
    var sym = String(newSymbol).toUpperCase().trim();
    if (!sym) return;
    if (sym === currentSymbol) return;

    if (_switchTimer) clearTimeout(_switchTimer);
    _pendingSymbol = sym;
    _switchTimer = setTimeout(function () {
        _switchTimer = null;
        var symToUse = _pendingSymbol;
        _pendingSymbol = null;
        if (!symToUse || symToUse === currentSymbol) return;

        var innerChart = getInnerChart();
        _skipDrawingDelete = true;
        if (innerChart) {
            innerChart.removeOverlay();
            var ids = overlayIdsBySymbol[currentSymbol];
            if (ids) {
                ids.forEach(function (id) { innerChart.removeOverlay(id); });
                overlayIdsBySymbol[currentSymbol] = [];
            }
        }
        _skipDrawingDelete = false;
        unsubscribeOverlays(currentSymbol);
        currentSymbol = symToUse;
        _pendingScrollToRealTime = true;
        chart.setSymbol(toSymbolInfo(symToUse));
        renderWatchlist();
        hideSymbolSearchPanel();
    }, 500);
}

function trySubscribeDrawings() {
    if (_drawingsWs && _drawingsWs.readyState === WebSocket.OPEN) {
        subscribeOverlays(currentSymbol);
    } else {
        setTimeout(trySubscribeDrawings, 10);
    }
}

if (typeof ChartIntercept !== 'undefined') {
    ChartIntercept.register(chartContainer, { onSymbolClick: showSymbolSearchPanel });
}

// ============ CHART UI HANDLERS ============
chartContainer.addEventListener('click', function (e) {
    var themeBtn = e.target.closest('[data-theme-btn]');
    if (themeBtn) {
        e.stopPropagation();
        e.preventDefault();
        var current = (typeof ChartSettingsModel !== 'undefined' && ChartSettingsModel.getThemeMode)
            ? ChartSettingsModel.getThemeMode() : 'dark';
        var next = current === 'dark' ? 'light' : 'dark';
        if (typeof ChartSettingsModel !== 'undefined') {
            ChartSettingsModel.setThemeMode(next);
            ChartSettingsModel.applyToCharts();
        } else {
            chart.setTheme(next);
        }
        return;
    }
    var candleBtn = e.target.closest('[data-candle-btn]');
    if (candleBtn) {
        e.stopPropagation();
        e.preventDefault();
        if (typeof ChartSettingsModel !== 'undefined' && typeof ThemePresets !== 'undefined') {
            var curId = ChartSettingsModel.getCandlePreset();
            var nextId = ThemePresets.getNextId(curId);
            ChartSettingsModel.setCandlePreset(nextId);
            ChartSettingsModel.applyToCharts();
            var preset = ThemePresets.getPreset(nextId);
            var inner = candleBtn.querySelector('.candle-btn-label');
            if (inner && preset) {
                inner.innerHTML = '<span style="color:' + preset.upColor + '">' + preset.code[0] + '</span>'
                    + '<span style="color:' + preset.downColor + '">' + preset.code[1] + '</span>';
            }
        }
        return;
    }
}, true);

// ============ CONTEXT MENU ============
(function () {
    var menu = null;
    var closeMenu = function () {
        if (!menu) return;
        menu.classList.remove('chart-context-menu-visible');
        menu.style.display = 'none';
        document.removeEventListener('click', closeMenu);
        document.removeEventListener('keydown', onEscape);
    };
    function onEscape(e) {
        if (e.key === 'Escape') closeMenu();
    }
    chartContainer.addEventListener('contextmenu', function (e) {
        e.preventDefault();
        if (!chart) return;
        if (!menu) {
            menu = document.createElement('div');
            menu.className = 'chart-context-menu';
            menu.innerHTML = '<div class="chart-context-menu-item" data-action="reset">Reset chart view</div>';
            menu.style.cssText = 'position:fixed;z-index:10000;background:#1e222d;border:1px solid #555;border-radius:6px;padding:4px 0;min-width:160px;box-shadow:0 4px 12px rgba(0,0,0,0.3);display:none;';
            var item = menu.querySelector('[data-action="reset"]');
            item.style.cssText = 'padding:8px 12px;cursor:pointer;font-family:Montserrat,sans-serif;font-size:13px;color:#ddd;';
            item.addEventListener('mouseenter', function () { item.style.background = '#2a2e39'; });
            item.addEventListener('mouseleave', function () { item.style.background = 'transparent'; });
            item.addEventListener('click', function (e) {
                e.stopPropagation();
                closeMenu();
                setTimeout(function () { resetChartView(); }, 0);
            });
            document.body.appendChild(menu);
        }
        menu.style.left = e.clientX + 'px';
        menu.style.top = e.clientY + 'px';
        menu.style.display = 'block';
        menu.classList.add('chart-context-menu-visible');
        setTimeout(function () {
            document.addEventListener('click', closeMenu);
            document.addEventListener('keydown', onEscape);
        }, 0);
    });
})();

var initialTheme = (typeof ChartSettingsModel !== 'undefined' && ChartSettingsModel.getThemeMode) ? ChartSettingsModel.getThemeMode() : 'dark';

// ============ OVERLAYS ============
// Override built-in Fibonacci Segment: 5 levels, percentage-only labels on right, segment-only lines, custom colors
registerOverlay({
    name: 'fibonacciSegment',
    totalStep: 3,
    needDefaultPointFigure: true,
    needDefaultXAxisFigure: true,
    needDefaultYAxisFigure: true,
    styles: {
        point: {
            color: '#555555',
            borderColor: '#555555',
            activeColor: '#555555',
            activeBorderColor: '#555555'
        },
        line: {
            color: '#555555'
        }
    },
    createPointFigures: function (params) {
        var coordinates = params.coordinates;
        if (!coordinates || coordinates.length < 2) return [];
        var c0 = coordinates[0];
        var c1 = coordinates[1];
        var x0 = c0.x;
        var x1 = c1.x;
        var rightX = (x0 > x1 ? x0 : x1) + 4;
        var dy = c0.y - c1.y;
        var levels = [
            { ratio: -0.618, label: '-61.80%', color: '#808080', size: 1 },
            { ratio: -0.272, label: '-27.20%', color: '#808080', size: 1 },
            { ratio: 0.5, label: '50.00%', color: '#E13255', size: 2 },
            { ratio: 0.618, label: '61.80%', color: '#FFC107', size: 1 },
            { ratio: 0.786, label: '78.60%', color: '#00FF00', size: 1 }
        ];
        var figures = [];
        levels.forEach(function (lev) {
            var y = c1.y + dy * lev.ratio;
            figures.push({
                type: 'line',
                attrs: [{ coordinates: [{ x: x0, y: y }, { x: x1, y: y }] }],
                styles: { color: lev.color, size: lev.size, style: 'solid', dashedValue: [] }
            });
            figures.push({
                type: 'text',
                ignoreEvent: true,
                attrs: { x: rightX, y: y, text: lev.label, baseline: 'bottom' },
                styles: { color: lev.color, backgroundColor: 'transparent' }
            });
        });
        return figures;
    }
});

// Override built-in Fibonacci Extension: levels 0.786, 1, 1.38, 1.618, 2; labels on right; same color as lines; no background
registerOverlay({
    name: 'fibonacciExtension',
    totalStep: 4,
    needDefaultPointFigure: true,
    needDefaultXAxisFigure: true,
    needDefaultYAxisFigure: true,
    styles: {
        point: {
            color: '#555555',
            borderColor: '#555555',
            activeColor: '#555555',
            activeBorderColor: '#555555'
        },
        line: {
            color: '#555555'
        }
    },
    createPointFigures: function (params) {
        var coordinates = params.coordinates;
        var overlay = params.overlay;
        var precision = params.precision;
        if (!coordinates || coordinates.length < 3 || !overlay || !overlay.points) return [];
        var points = overlay.points;
        var o = points[1].value - points[0].value;
        var s = coordinates[1].y - coordinates[0].y;
        var levels = [0.786, 1, 1.38, 1.618, 2];
        var rightX = (coordinates[2].x > coordinates[1].x ? coordinates[2].x : coordinates[1].x) + 4;
        var lineColor = '#555555';
        var r = [];
        var a = [];
        levels.forEach(function (u) {
            var y = coordinates[2].y + s * u;
            var p = (points[2].value + o * u).toFixed(precision.price);
            r.push({
                coordinates: [{ x: coordinates[1].x, y: y }, { x: coordinates[2].x, y: y }]
            });
            a.push({
                x: rightX,
                y: y,
                text: u + ' (' + p + ')',
                baseline: 'bottom'
            });
        });
        return [
            {
                type: 'line',
                attrs: { coordinates: coordinates },
                styles: { style: 'dashed' }
            },
            {
                type: 'line',
                attrs: r,
                styles: { color: lineColor }
            },
            {
                type: 'text',
                ignoreEvent: true,
                attrs: a,
                styles: { color: lineColor, backgroundColor: 'transparent' }
            }
        ];
    }
});

// ============ CHART HELPERS ============
var chart;

function getInnerChart() {
    if (!chartContainer || !chart) return null;
    var inner = chartContainer.querySelector('[k-line-chart-id]');
    if (!inner) return null;
    var id = inner.getAttribute('k-line-chart-id');
    if (!id) return null;
    inner.id = id;
    return klinechartsInit(inner) || null;
}

function resetChartView() {
    var chartInstance = getInnerChart();
    if (!chartInstance) return;
    
    // Wait a bit to ensure data is loaded before resetting Y-axis
    setTimeout(function() {
        // Find the Y-axis canvas element and simulate double-click to reset range
        var chartDom = chartContainer.querySelector('.klinecharts-pro');
        if (chartDom) {
            var canvases = chartDom.querySelectorAll('canvas');
            if (canvases && canvases.length > 0) {
                var mainCanvas = canvases[0];
                var rect = mainCanvas.getBoundingClientRect();
                // Y-axis is on the right side - use 95% of width
                var yAxisX = rect.left + (rect.width * 0.95);
                var yAxisY = rect.top + (rect.height * 0.5);
                
                var dblClickEvent = new MouseEvent('dblclick', {
                    bubbles: true,
                    cancelable: true,
                    view: window,
                    clientX: yAxisX,
                    clientY: yAxisY
                });
                mainCanvas.dispatchEvent(dblClickEvent);
            }
        }
        
        // Also try zoom reset as fallback
        if (typeof chartInstance.zoomAtDataIndex === 'function' && typeof chartInstance.getDataList === 'function') {
            var dataList = chartInstance.getDataList();
            if (dataList && dataList.length > 0) {
                chartInstance.zoomAtDataIndex(1.0, dataList.length - 1, 0);
            }
        }
        
        // Finally scroll to real time
        if (typeof chartInstance.scrollToRealTime === 'function') {
            chartInstance.scrollToRealTime(0);
        }
    }, 500);  // Wait 500ms for data to load
}

function serializeOverlay(overlay) {
    if (!overlay) return null;
    return {
        id: overlay.id,
        name: overlay.name,
        points: overlay.points,
        extendData: overlay.extendData,
        styles: overlay.styles,
        groupId: overlay.groupId,
        lock: overlay.lock != null ? overlay.lock : false,
        visible: overlay.visible != null ? overlay.visible : true,
        zLevel: overlay.zLevel != null ? overlay.zLevel : 0
    };
}

// ============ DRAWINGS ============
function subscribeOverlays(symbol) {
    if (!_drawingsWs || _drawingsWs.readyState !== WebSocket.OPEN) return;
    _drawingsWs.send(JSON.stringify({ type: 'subscribe_drawings', symbol: symbol || currentSymbol }));
}

function unsubscribeOverlays(symbol) {
    if (!_drawingsWs || _drawingsWs.readyState !== WebSocket.OPEN) return;
    _drawingsWs.send(JSON.stringify({ type: 'unsubscribe_drawings', symbol: symbol || currentSymbol }));
}

function onDrawingsSnapshot(data) {
    var symbol = data && data.symbol;
    var overlays = data && data.overlays;
    if (!symbol || !Array.isArray(overlays)) return;
    var innerChart = getInnerChart();
    if (!innerChart) return;
    overlayIdsBySymbol[symbol] = overlayIdsBySymbol[symbol] || [];
    _loadingDrawingsSnapshot = true;
    overlays.forEach(function (item) {
        item.onRemoved = onOverlayRemoved;
        var res = innerChart.createOverlay(item);
        if (res != null) {
            if (Array.isArray(res)) {
                res.forEach(function (id) { if (id) overlayIdsBySymbol[symbol].push(id); });
            } else {
                overlayIdsBySymbol[symbol].push(res);
            }
        }
    });
    _loadingDrawingsSnapshot = false;
}

function onOverlayRemoved(ev) {
    var overlay = ev.overlay;
    if (!overlay || !overlay.id) return;
    if (_skipDrawingDelete) return;
    var sym = currentSymbol;
    var ids = overlayIdsBySymbol[sym];
    if (ids) {
        var idx = ids.indexOf(overlay.id);
        if (idx !== -1) ids.splice(idx, 1);
    }
    if (_drawingsWs && _drawingsWs.readyState === WebSocket.OPEN) {
        _drawingsWs.send(JSON.stringify({
            type: 'drawing_delete',
            symbol: sym,
            overlayId: overlay.id
        }));
    }
}

// ============ CHART INIT ============
function initChart(watermarkNode) {
    var options = {
        container: chartContainer,
        theme: initialTheme,
        symbol: {
            ticker: 'BTCUSDT',
            shortName: 'BTCUSDT',
            exchange: 'BINANCE',
            market: 'crypto',
            priceCurrency: 'usd'
        },
        period: { multiplier: 15, timespan: 'minute', text: '15m' },
        periods: [
            { multiplier: 1, timespan: 'minute', text: '1m' },
            { multiplier: 5, timespan: 'minute', text: '5m' },
            { multiplier: 15, timespan: 'minute', text: '15m' },
            { multiplier: 1, timespan: 'hour', text: '1H' },
            { multiplier: 2, timespan: 'hour', text: '2H' },
            { multiplier: 4, timespan: 'hour', text: '4H' },
            { multiplier: 1, timespan: 'day', text: 'D' },
            { multiplier: 1, timespan: 'week', text: 'W' },
            { multiplier: 1, timespan: 'month', text: 'M' }
        ],
        datafeed: new BinanceDatafeed(),
        locale: 'en-US',
        styles: {
            xAxis: { tickText: { family: 'Montserrat', weight: 600 } },
            yAxis: { tickText: { family: 'Montserrat', weight: 600 } },
            crosshair: {
                horizontal: { text: { family: 'Montserrat', weight: 600 } },
                vertical: { text: { family: 'Montserrat', weight: 600 } }
            },
            candle: {
                tooltip: {
                    title: { family: 'Montserrat', weight: 600 },
                    legend: { family: 'Montserrat', weight: 600 }
                },
                priceMark: {
                    high: { textFamily: 'Montserrat', textWeight: '600' },
                    low: { textFamily: 'Montserrat', textWeight: '600' },
                    last: { text: { family: 'Montserrat', weight: 600 } }
                }
            },
            indicator: {
                tooltip: {
                    title: { family: 'Montserrat', weight: 600 },
                    legend: { family: 'Montserrat', weight: 600 }
                },
                lastValueMark: { text: { family: 'Montserrat', weight: 600 } }
            }
        }
    };
    if (watermarkNode) options.watermark = watermarkNode;
    chart = new KLineChartPro(options);

    // Scroll to real time only after symbol switch (when new data is loaded), not on every real-time update
    setTimeout(function () {
        var innerChart = getInnerChart();
        if (innerChart && typeof innerChart.subscribeAction === 'function') {
            innerChart.subscribeAction('onDataReady', function () {
                if (_pendingScrollToRealTime && innerChart) {
                    // Data-driven approach: check if data is actually loaded
                    var attemptReset = function(attempt) {
                        if (attempt > 10) {  // Max 10 attempts = 500ms
                            _pendingScrollToRealTime = false;
                            return;
                        }
                        
                        // Check if data list has content
                        var dataList = typeof innerChart.getDataList === 'function' ? innerChart.getDataList() : null;
                        if (!dataList || dataList.length === 0) {
                            // No data yet, try again in 50ms
                            setTimeout(function() { attemptReset(attempt + 1); }, 50);
                            return;
                        }
                        
                        // Data is loaded, trigger Y-axis reset immediately
                        var chartDom = chartContainer.querySelector('.klinecharts-pro');
                        if (chartDom) {
                            var canvases = chartDom.querySelectorAll('canvas');
                            if (canvases && canvases.length > 0) {
                                var mainCanvas = canvases[0];
                                var rect = mainCanvas.getBoundingClientRect();
                                // Y-axis is on the right side - use 95% of width
                                var yAxisX = rect.left + (rect.width * 0.95);
                                var yAxisY = rect.top + (rect.height * 0.5);
                                
                                var dblClickEvent = new MouseEvent('dblclick', {
                                    bubbles: true,
                                    cancelable: true,
                                    view: window,
                                    clientX: yAxisX,
                                    clientY: yAxisY
                                });
                                mainCanvas.dispatchEvent(dblClickEvent);
                            }
                        }
                        
                        // Scroll to real time
                        if (typeof innerChart.scrollToRealTime === 'function') {
                            innerChart.scrollToRealTime(0);
                        }
                        
                        _pendingScrollToRealTime = false;
                    };
                    
                    // Start checking immediately (no arbitrary delay)
                    attemptReset(1);
                }
            });
        }
    }, 100);

    if (typeof ChartSettingsModel !== 'undefined') {
        ChartSettingsModel.setChartRefs({ mainChart: chart });
        ChartSettingsModel.applyToCharts();
    }
    (function () {
        var resizeTimer;
        window.addEventListener('resize', function () {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(function () {
                if (chart && typeof chart.resize === 'function') chart.resize();
            }, 200);
        });
    })();
    initWatchlist();
    setTimeout(injectThemeToggle, 0);
    setTimeout(injectCandleToggle, 0);
    setTimeout(injectSaveDrawingsButton, 0);
    setTimeout(removeToolbarTextForScreenshotAndFullScreen, 50);
    setTimeout(function installDrawingsFeed() {
        var innerChart = getInnerChart();
        if (!innerChart) return;
        _onDrawingsSnapshot = onDrawingsSnapshot;
        var origCreate = innerChart.createOverlay;
        if (typeof origCreate !== 'function') return;
        innerChart.createOverlay = function (value, paneId) {
            var result = origCreate.call(innerChart, value, paneId);
            if (result != null) {
                var ids = Array.isArray(result) ? result : [result];
                ids.forEach(function (id) {
                    if (id) {
                        overlayIdsBySymbol[currentSymbol] = overlayIdsBySymbol[currentSymbol] || [];
                        overlayIdsBySymbol[currentSymbol].push(id);
                        innerChart.overrideOverlay({ id: id, onRemoved: onOverlayRemoved });
                        if (!_loadingDrawingsSnapshot && _drawingsWs && _drawingsWs.readyState === WebSocket.OPEN) {
                            var ov = innerChart.getOverlayById(id);
                            if (ov) {
                                _drawingsWs.send(JSON.stringify({
                                    type: 'drawing_create',
                                    symbol: currentSymbol,
                                    overlay: serializeOverlay(ov)
                                }));
                            }
                        }
                    }
                });
            }
            return result;
        };
        trySubscribeDrawings();
    }, 0);
}

// ============ CHART BOOTSTRAP ============
var watermarkImg = new Image();
watermarkImg.style.maxWidth = '100%';
watermarkImg.style.maxHeight = '100%';
watermarkImg.style.objectFit = 'contain';
watermarkImg.style.opacity = '0.15';
watermarkImg.style.pointerEvents = 'none';
watermarkImg.onload = function () { initChart(watermarkImg); };
watermarkImg.onerror = function () { initChart(null); };
watermarkImg.src = watermarkUrl;

// ============ TOOLBAR INJECTION ============
function injectThemeToggle() {
    var periodBar = chartContainer.querySelector('.klinecharts-pro-period-bar');
    if (!periodBar) return;
    var tools = periodBar.querySelectorAll('.item.tools');
    if (!tools.length) return;
    var lastTool = tools[tools.length - 1];
    var themeBtn = document.createElement('div');
    themeBtn.className = 'item tools toolbar-theme-btn';
    themeBtn.style.cursor = 'pointer';
    themeBtn.style.display = 'flex';
    themeBtn.style.alignItems = 'center';
    themeBtn.style.justifyContent = 'center';
    themeBtn.setAttribute('data-theme-btn', 'true');
    var img = document.createElement('img');
    img.src = themeIconUrl;
    img.alt = 'Theme';
    img.title = 'Switch theme';
    img.style.width = '20px';
    img.style.height = '20px';
    img.style.objectFit = 'contain';
    themeBtn.appendChild(img);
    lastTool.parentNode.insertBefore(themeBtn, lastTool.nextSibling);
}

function injectCandleToggle() {
    var periodBar = chartContainer.querySelector('.klinecharts-pro-period-bar');
    if (!periodBar) return;
    var tools = periodBar.querySelectorAll('.item.tools');
    if (!tools.length) return;
    var lastTool = tools[tools.length - 1];
    var candleBtn = document.createElement('div');
    candleBtn.className = 'item tools';
    candleBtn.style.cursor = 'pointer';
    candleBtn.setAttribute('data-candle-btn', 'true');
    var preset = (typeof ChartSettingsModel !== 'undefined' && typeof ThemePresets !== 'undefined')
        ? ThemePresets.getPreset(ChartSettingsModel.getCandlePreset())
        : { code: 'GR', upColor: '#2DC08E', downColor: '#F92855' };
    var label = document.createElement('span');
    label.className = 'candle-btn-label';
    label.innerHTML = '<span style="color:' + preset.upColor + '">' + preset.code[0] + '</span>'
        + '<span style="color:' + preset.downColor + '">' + preset.code[1] + '</span>';
    candleBtn.appendChild(label);
    lastTool.parentNode.insertBefore(candleBtn, lastTool.nextSibling);
}

function injectSaveDrawingsButton() {
    var periodBar = chartContainer.querySelector('.klinecharts-pro-period-bar');
    if (!periodBar) return;
    var tools = periodBar.querySelectorAll('.item.tools');
    if (!tools.length) return;
    var lastTool = tools[tools.length - 1];
    var saveBtn = document.createElement('div');
    saveBtn.className = 'item tools toolbar-save-btn';
    saveBtn.style.cursor = 'pointer';
    saveBtn.style.display = 'flex';
    saveBtn.style.alignItems = 'center';
    saveBtn.style.justifyContent = 'center';
    saveBtn.title = 'Save drawings';
    var img = document.createElement('img');
    img.src = saveIconUrl;
    img.alt = 'Save';
    img.style.width = '20px';
    img.style.height = '20px';
    img.style.objectFit = 'contain';
    saveBtn.appendChild(img);
    saveBtn.addEventListener('click', function () {
        saveDrawings();
    });
    lastTool.parentNode.insertBefore(saveBtn, lastTool.nextSibling);
}

function saveDrawings() {
    if (!_drawingsWs || _drawingsWs.readyState !== WebSocket.OPEN) return;
    var innerChart = getInnerChart();
    if (!innerChart || !currentSymbol) return;
    var ids = overlayIdsBySymbol[currentSymbol] || [];
    var arr = [];
    for (var i = 0; i < ids.length; i++) {
        var ov = innerChart.getOverlayById(ids[i]);
        if (ov) arr.push(serializeOverlay(ov));
    }
    _drawingsWs.send(JSON.stringify({
        type: 'drawing_save',
        symbol: currentSymbol,
        overlays: arr
    }));
}
function removeToolbarTextForScreenshotAndFullScreen() {
    var periodBar = chartContainer.querySelector('.klinecharts-pro-period-bar');
    if (!periodBar) return;
    var tools = periodBar.querySelectorAll('.item.tools');
    var texts = [
        'Screenshot',
        'Full screen',
        'Exit full screen',
        'Full Screen',
        'Exit Full Screen',
        'Exit',
        'Indicator',
        'Timezone',
        'Setting'
    ];
    tools.forEach(function (el) {
        var t = (el.textContent || '').trim();
        if (texts.indexOf(t) >= 0) {
            var span = el.querySelector('span');
            if (span) span.textContent = '';
            else el.textContent = '';
        }
    });
}
// ============ SYMBOL SEARCH INPUT ============
if (symbolSearchInput) {
    symbolSearchInput.addEventListener('input', function () {
        var query = symbolSearchInput.value;
        searchSymbolsForUI(query).then(renderSymbolList);
    });
}
