/**
 * Settings panel view: builds overlay + panel DOM, injects into container, binds events.
 * Exposes show() and hide(). Calls ChartSettingsModel for state and apply.
 */
(function (global) {
    var container = null;
    var overlay = null;
    var panel = null;

    function buildPanel() {
        var Model = global.ChartSettingsModel;
        var Presets = global.ThemePresets;
        if (!Model || !Presets) return;

        var overlayEl = document.createElement('div');
        overlayEl.id = 'settings-panel-overlay';
        overlayEl.className = 'settings-panel-overlay';
        overlayEl.setAttribute('aria-hidden', 'true');

        var panelEl = document.createElement('div');
        panelEl.id = 'settings-panel';
        panelEl.className = 'settings-panel';
        panelEl.setAttribute('aria-hidden', 'true');

        var header = document.createElement('div');
        header.className = 'settings-panel-header';
        var title = document.createElement('span');
        title.className = 'settings-panel-title';
        title.textContent = 'Settings';
        var closeBtn = document.createElement('button');
        closeBtn.type = 'button';
        closeBtn.className = 'settings-panel-close';
        closeBtn.id = 'settings-panel-close';
        closeBtn.title = 'Close';
        closeBtn.textContent = '\u00D7';
        header.appendChild(title);
        header.appendChild(closeBtn);
        panelEl.appendChild(header);

        var body = document.createElement('div');
        body.className = 'settings-panel-body';

        var themeSection = document.createElement('section');
        themeSection.className = 'settings-section';
        var themeLabel = document.createElement('div');
        themeLabel.className = 'settings-section-label';
        themeLabel.textContent = 'Theme';
        var themeModeWrap = document.createElement('div');
        themeModeWrap.className = 'settings-theme-mode';
        var darkBtn = document.createElement('button');
        darkBtn.type = 'button';
        darkBtn.className = 'settings-mode-btn';
        darkBtn.setAttribute('data-mode', 'dark');
        darkBtn.id = 'settings-mode-dark';
        darkBtn.textContent = 'Dark';
        var lightBtn = document.createElement('button');
        lightBtn.type = 'button';
        lightBtn.className = 'settings-mode-btn';
        lightBtn.setAttribute('data-mode', 'light');
        lightBtn.id = 'settings-mode-light';
        lightBtn.textContent = 'Light';
        themeModeWrap.appendChild(darkBtn);
        themeModeWrap.appendChild(lightBtn);
        themeSection.appendChild(themeLabel);
        themeSection.appendChild(themeModeWrap);
        body.appendChild(themeSection);

        var candleSection = document.createElement('section');
        candleSection.className = 'settings-section';
        var candleLabel = document.createElement('div');
        candleLabel.className = 'settings-section-label';
        candleLabel.textContent = 'Candle colors';
        var candleWrap = document.createElement('div');
        candleWrap.className = 'settings-candle-presets';
        candleWrap.id = 'settings-candle-presets';
        var ids = Presets.getIds ? Presets.getIds() : Object.keys(Presets).filter(function (k) { return typeof Presets[k] === 'object' && Presets[k].name; });
        ids.forEach(function (id) {
            var p = Presets.getPreset ? Presets.getPreset(id) : Presets[id];
            if (!p || !p.name) return;
            var btn = document.createElement('button');
            btn.type = 'button';
            btn.className = 'settings-candle-item';
            btn.setAttribute('data-theme', id);
            var swatch = document.createElement('span');
            swatch.className = 'swatch';
            swatch.style.background = 'linear-gradient(180deg,' + (p.upColor || '') + ' 50%,' + (p.downColor || '') + ' 50%)';
            var nameSpan = document.createElement('span');
            nameSpan.textContent = p.name;
            btn.appendChild(swatch);
            btn.appendChild(nameSpan);
            btn.addEventListener('click', function () {
                Model.setCandlePreset(id);
                Model.applyToCharts();
                updateSelection();
            });
            candleWrap.appendChild(btn);
        });
        candleSection.appendChild(candleLabel);
        candleSection.appendChild(candleWrap);
        body.appendChild(candleSection);

        panelEl.appendChild(body);

        overlayEl.addEventListener('click', hide);
        closeBtn.addEventListener('click', hide);
        darkBtn.addEventListener('click', function () {
            Model.setThemeMode('dark');
            Model.applyToCharts();
            panelEl.classList.remove('light');
            updateSelection();
        });
        lightBtn.addEventListener('click', function () {
            Model.setThemeMode('light');
            Model.applyToCharts();
            panelEl.classList.add('light');
            updateSelection();
        });

        document.addEventListener('keydown', function (e) {
            if (e.key === 'Escape' && panelEl.classList.contains('show')) hide();
        });

        function updateSelection() {
            var mode = Model.getThemeMode();
            panelEl.classList.toggle('light', mode === 'light');
            [darkBtn, lightBtn].forEach(function (btn) {
                btn.classList.toggle('selected', btn.getAttribute('data-mode') === mode);
            });
            var presetId = Model.getCandlePreset();
            candleWrap.querySelectorAll('.settings-candle-item').forEach(function (el) {
                el.classList.toggle('selected', el.getAttribute('data-theme') === presetId);
            });
        }

        overlay = overlayEl;
        panel = panelEl;
        panel._updateSelection = updateSelection;
    }

    function show() {
        if (!container) container = document.getElementById('settings-panel-container');
        if (!container) return;
        if (!overlay || !panel) {
            buildPanel();
            if (overlay && panel) {
                if (!overlay.parentNode) document.body.appendChild(overlay);
                if (!panel.parentNode) document.body.appendChild(panel);
            }
        }
        if (overlay) {
            overlay.classList.add('show');
            overlay.setAttribute('aria-hidden', 'false');
        }
        if (panel) {
            panel.classList.add('show');
            panel.setAttribute('aria-hidden', 'false');
            panel.classList.toggle('light', global.ChartSettingsModel.getThemeMode() === 'light');
            if (panel._updateSelection) panel._updateSelection();
        }
    }

    function hide() {
        if (overlay) {
            overlay.classList.remove('show');
            overlay.setAttribute('aria-hidden', 'true');
        }
        if (panel) {
            panel.classList.remove('show');
            panel.setAttribute('aria-hidden', 'true');
        }
    }

    function bindSettingsButton() {
        var btn = document.getElementById('settings-button');
        if (btn) btn.addEventListener('click', function (e) { e.stopPropagation(); show(); });
    }

    buildPanel();
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', bindSettingsButton);
    } else {
        bindSettingsButton();
    }

    global.SettingsPanelView = {
        show: show,
        hide: hide,
        bindSettingsButton: bindSettingsButton
    };
})(typeof window !== 'undefined' ? window : this);
