# V3 frontend: build workflow and custom overlay method

## How it works

- **Frontend** is a Vite app: source in `frontend/`, built output in `frontend/dist/`.
- **Backend** serves the built app when it exists: if `frontend/dist` is present it serves from there; otherwise it serves the raw `frontend/` folder.
- **Chart and overlays** come from `node_modules` (`klinecharts` and `@klinecharts/pro`). We call `registerOverlay` from `klinecharts` **before** creating the chart so the same registry is used and overrides (e.g. Fibonacci Segment) apply.

## Workflow

### Develop

- Edit files under `frontend/` (e.g. `app.js`, `index.html`, legacy scripts).
- Optional: run `npm run dev` in `frontend/` for live reload; the dev server resolves `klinecharts` and `@klinecharts/pro` from `node_modules`.

### Build

- In `frontend/`: **`npm run build`**.
- Produces `frontend/dist/` (index.html + assets/ with one JS bundle, CSS, fonts, images).
- No "can't be bundled without type=module" warnings if all app scripts are imported from `app.js`.

### Run

- Start the backend (e.g. run `back_end.py`).
- Backend serves `frontend/dist` when that folder exists, so the browser gets the built app.
- Open the URL the backend prints (e.g. `http://127.0.0.1:8765/index.html`).

### Verify

- Chart loads; use the **Fibonacci Segment** tool (two points). You should see the custom 5 levels, segment-only lines, percentage-only labels, and colors.

## Method: custom / override overlays

### Import from packages, not CDN

In `app.js` use:

```js
import { registerOverlay } from 'klinecharts';
import { KLineChartPro } from '@klinecharts/pro';
```

So the bundle uses a single klinecharts instance and overlay registry.

### Register before the chart

Call `registerOverlay({ name: '...', totalStep, needDefaultPointFigure, createPointFigures, ... })` **before** `new KLineChartPro(...)`. Override the built-in by using the same `name` (e.g. `'fibonacciSegment'`).

### Use the real API

- **Template:** `totalStep` (singular), `createPointFigures` (not `createFigures`).
- **Callback** receives `{ coordinates, overlay, bounding, xAxis, yAxis }`. Use `coordinates` for pixel positions and `overlay.points` for price data.
- Compute level Y from coordinates, e.g. `y = coordinates[1].y + (coordinates[0].y - coordinates[1].y) * ratio`.
- Return figures with `type`, `attrs`, and optional `styles` / `ignoreEvent`.

### Bundle all app scripts

Import any script that must be in the app from `app.js` (e.g. `import './themePresets.js';`) and remove their `<script src="...">` tags from `index.html` so Vite bundles them and the build stays clean.

## Key files

| Path | Role |
|------|------|
| `frontend/app.js` | Entry: imports klinecharts, Pro, legacy scripts; registers overlay; creates chart. |
| `frontend/index.html` | Single module script: `./app.js` (Vite inlines the built bundle). |
| `frontend/vite.config.js` | Vite config: root, `outDir: 'dist'`. |
| `frontend/dist/` | Built output; backend serves this when present. |
| `backend/back_end.py` | `_start_frontend_server()`: uses `frontend/dist` if it exists, else `frontend`. |

## Do / don't

- **Do** run `npm run build` before expecting the backend to serve the latest app.
- **Do** use `registerOverlay` from `klinecharts` and create the chart with `KLineChartPro` from `@klinecharts/pro` in the same bundled app.
- **Don't** use CDN imports for klinecharts/Pro in the built app if you rely on overlay overrides.
- **Don't** rely on the backend serving source `frontend/` for production-like behavior; it's for fallback when `dist` doesn't exist.
